<?php

namespace App\Http\Controllers;

use App\Models\Refund;
use App\Models\Payment;
use App\Models\Transaction;
use App\Services\PaymentHandler;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class RefundsController extends Controller
{
    public function index(): JsonResponse
    {
        $refunds = Refund::with(['payment', 'transaction'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $refunds,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $refund = Refund::with(['payment', 'transaction'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $refund,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Payment_id' => 'required|exists:Payments,Payment_id',
            'Amount' => 'required|numeric|min:0.01',
            'Currency' => 'required|string|size:3',
            'Reason' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $payment = Payment::findOrFail($request->Payment_id);
        if ($payment->Payment_status !== 'completed') {
            return response()->json([
                'status' => 'error',
                'message' => 'Only completed payments can be refunded.',
            ], 400);
        }

        $refund = Refund::create([
            'Payment_id' => $request->Payment_id,
            'Amount' => $request->Amount,
            'Currency' => $request->Currency,
            'Reason' => $request->Reason,
            'Status' => 'requested',
        ]);

        try {
            $paymentHandler = new PaymentHandler($payment->Payment_gateway);
            $response = $paymentHandler->processRefund($payment, $refund);
            $refund->update([
                'Status' => 'processed',
                'Gateway_response' => json_encode($response),
                'Processed_at' => now(),
            ]);

            if ($payment->Transaction_id) {
                $transaction = Transaction::where('Payment_id', $payment->Payment_id)->first();
                if ($transaction) {
                    $transaction->update(['Status' => 'refunded']);
                }
            }

            return response()->json([
                'status' => 'success',
                'data' => $refund,
                'message' => 'Refund processed successfully.',
            ], 201);
        } catch (\Exception $e) {
            Log::error('Refund Error: ' . $e->getMessage());
            $refund->update(['Status' => 'failed', 'Gateway_response' => $e->getMessage()]);
            return response()->json([
                'status' => 'error',
                'message' => 'Refund failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        $refund = Refund::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Amount' => 'sometimes|numeric|min:0.01',
            'Currency' => 'sometimes|string|size:3',
            'Reason' => 'sometimes|string',
            'Status' => 'sometimes|in:requested,processed,failed',
            'Gateway_response' => 'sometimes|nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $refund->update($request->only([
            'Amount',
            'Currency',
            'Reason',
            'Status',
            'Gateway_response',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $refund,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $refund = Refund::findOrFail($id);
        $refund->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Refund deleted successfully',
        ], 200);
    }
}
